/* --COPYRIGHT--,BSD
 * Copyright (c) 2015, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//******************************************************************************
//  Multiple time-base example
//
//  ACLK = 32kHz, MCLK = SMCLK = TBCLK = 25MHz
//
//  As coded and with TBCLK = ~25MHz, toggle rates are:
//  P4.0= CCR0 = ~25MHz/(2*200)  = ~62.5kHz   
//  P4.1= CCR1 = ~25MHz/(2*150)  = ~83kHz    
//  P4.2= CCR2 = ~25MHz/(2*2000) = ~6.25kHz  
//  P4.3= CCR3 = ~25MHz/(2*2500) = ~5.0kHz   
//  P4.4= CCR4 = ~25MHz/(2*3000) = ~4.17kHz  
//  P4.5= CCR5 = ~25MHz/(2*3500) = ~3.57kHz  
//  P4.6= CCR6 = ~25MHz/(2*4000) = ~3.125kHz 
//  P1.0= overflow = ~25MHz/(2*65536) = 190.7Hz
//
//               MSP430F5529
//            -------------------
//        /|\|                   |
//         | |                   |
//         --|RST                |
//           |                   |
//           |         P4.0/TB.0|--> CCR0 ~62.5kHz  
//           |         P4.1/TB.1|--> CCR1 ~83kHz    
//           |         P4.2/TB.2|--> CCR2 ~6.25kHz  
//           |         P4.3/TB.3|--> CCR3 ~5.0kHz   
//           |         P4.4/TB.4|--> CCR4 ~4.17kHz  
//           |         P4.5/TB.5|--> CCR5 ~3.57kHz  
//           |         P4.6/TB.6|--> CCR6 ~3.125kHz 
//           |               P1.0|--> Overflow/software
//           |               P1.0|--> LED
//
// Katie Enderle
// July 2011
//******************************************************************************

#include "msp430.h"
#include "HAL_PMM.h"

// Port4 Port Mapping definitions
const unsigned char P4mapping[8] = {
                                      PM_TB0CCR0A,
                                      PM_TB0CCR1A,             
                                      PM_TB0CCR2A,             
                                      PM_TB0CCR3A,             
                                      PM_TB0CCR4A,             
                                      PM_TB0CCR5A,             
                                      PM_TB0CCR6A,
                                      PM_NONE     };
// Function Definition
void Port_Mapping(void);

void main(void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop WDT
  
  SetVCore (3);  
  
  UCSCTL3 = SELREF_2;                       // Set DCO FLL reference = REFO
  UCSCTL4 |= SELA_2;                        // Set ACLK = REFO

  __bis_SR_register(SCG0);                  // Disable the FLL control loop
  UCSCTL0 = 0x0000;                         // Set lowest possible DCOx, MODx
  UCSCTL1 = DCORSEL_7;                      // Select DCO range 50MHz operation
  UCSCTL2 = FLLD_1 + 762;                   // Set DCO Multiplier for 25MHz
                                            // (N + 1) * FLLRef = Fdco
                                            // (762 + 1) * 32768 = 25MHz
                                            // Set FLL Div = fDCOCLK/2
  __bic_SR_register(SCG0);                  // Enable the FLL control loop

  // Worst-case settling time for the DCO when the DCO range bits have been
  // changed is n x 32 x 32 x f_MCLK / f_FLL_reference. See UCS chapter in 5xx
  // UG for optimization.
  // 32 x 32 x 25 MHz / 32,768 Hz ~ 780k MCLK cycles for DCO to settle
  __delay_cycles(782000);

  // Loop until XT1,XT2 & DCO stabilizes - In this case only DCO has to stabilize
  do
  {
    UCSCTL7 &= ~(XT2OFFG + XT1LFOFFG + DCOFFG);
                                            // Clear XT2,XT1,DCO fault flags
    SFRIFG1 &= ~OFIFG;                      // Clear fault flags
  }while (SFRIFG1&OFIFG);                   // Test oscillator fault flag
  
  Port_Mapping();
  
  // Setup Port Pins              
  P4DIR |= 0xFF;                            // P4.0 - P4.7 output
  P4SEL |= 0xFF;                            // P4.0 - P4.6 Port Map functions
  P1DIR |= 0x01;                            // P1.0 - Outputs
  
  //Set up Timer B in continuous mode for multiple time base PWMs
  TBCCTL0 = OUTMOD_4 + CCIE;               // CCR0 toggle, interrupt enabled
  TBCCTL1 = OUTMOD_4 + CCIE;               // CCR1 toggle, interrupt enabled
  TBCCTL2 = OUTMOD_4 + CCIE;               // CCR2 toggle, interrupt enabled
  TBCCTL3 = OUTMOD_4 + CCIE;               // CCR3 toggle, interrupt enabled
  TBCCTL4 = OUTMOD_4 + CCIE;               // CCR4 toggle, interrupt enabled
  TBCCTL5 = OUTMOD_4 + CCIE;               // CCR5 toggle, interrupt enabled
  TBCCTL6 = OUTMOD_4 + CCIE;               // CCR5 toggle, interrupt enabled
  TBCTL = TBSSEL_2 + MC_2 + TBCLR + TBIE;  // SMCLK, contmode, clear TAR,
                                           // interrupt enabled

  __bis_SR_register(LPM0_bits + GIE);       // Enter LPM3, interrupts enabled
  __no_operation();                         // For debugger
}

/*Multiple timebase PWMs
 *Period is determined by the sum of the two TBCCRx offset values
 *Duty cycle is determined by the ration of the high/low TBCCRx value
 *to the total period.
 */

// Timer B0 interrupt service routine
#pragma vector=TIMERB0_VECTOR
__interrupt void TIMERB0_ISR (void)
{
  TBCCR0 += 200;                            // reload period
}

// Timer_B1 Interrupt Vector (TBIV) handler
#pragma vector=TIMERB1_VECTOR
__interrupt void TIMERB1_ISR(void)
{
  switch(__even_in_range(TBIV,14))
  {
    case 0: break;                  
    case 2:  TB0CCR1 += 150;                // reload period
             break;
    case 4:  TB0CCR2 += 2000;               // reload period
             break;
    case 6:  TB0CCR3 += 2500;               // reload period
             break;                         
    case 8:  TB0CCR4 += 3000;               // reload period
             break;                         
    case 10: TB0CCR5 += 3500;               // reload period
             break;                         
    case 12: TB0CCR6 += 4000;               // reload period
             break;                         
    case 14: P1OUT ^= 0x01;                 // overflow
             break;
    default: break;
 }
}

void Port_Mapping(void)
{
  unsigned char i;
  volatile unsigned char *ptr;
  __disable_interrupt();                    // Disable Interrupts before altering Port Mapping registers
  PMAPPWD = 0x02D52;                        // Enable Write-access to modify port mapping registers
  
  #ifdef PORT_MAP_RECFG                     
  PMAPCTL = PMAPRECFG;                      // Allow reconfiguration during runtime
  #endif  
  
  ptr = &P4MAP0;
  for(i=0;i<8;i++)
  {
    *ptr = P4mapping[i];
    ptr++;
  } 
  
  PMAPPWD = 0;                              // Disable Write-Access to modify port mapping registers
  #ifdef PORT_MAP_EINT
  __enable_interrupt();                     // Re-enable all interrupts
  #endif  
  
}
